<?php
/**
 * @version     2.0.0
 * @package     com_draughts
 * @copyright   Copyright (C) 2015. All rights reserved.
 * @license     GNU General Public License version 2 or later; see LICENSE.txt
 * @author      Galaa <contact@galaa.mn> - http://galaa.mn
 */

// No direct access.
defined('_JEXEC') or die;

jimport('joomla.application.component.modeladmin');

/**
 * Draughts model.
 */
class DraughtsModelGame extends JModelAdmin
{
	/**
	 * @var		string	The prefix to use with controller messages.
	 * @since	1.6
	 */
	protected $text_prefix = 'COM_DRAUGHTS';


	/**
	 * Returns a reference to the a Table object, always creating it.
	 *
	 * @param	type	The table type to instantiate
	 * @param	string	A prefix for the table class name. Optional.
	 * @param	array	Configuration array for model. Optional.
	 * @return	JTable	A database object
	 * @since	1.6
	 */
	public function getTable($type = 'Game', $prefix = 'DraughtsTable', $config = array())
	{
		return JTable::getInstance($type, $prefix, $config);
	}

	/**
	 * Method to get the record form.
	 *
	 * @param	array	$data		An optional array of data for the form to interogate.
	 * @param	boolean	$loadData	True if the form is to load its own data (default case), false if not.
	 * @return	JForm	A JForm object on success, false on failure
	 * @since	1.6
	 */
	public function getForm($data = array(), $loadData = true)
	{
		// Initialise variables.
		$app = JFactory::getApplication();

		// Get the form.
		$form = $this->loadForm('com_draughts.game', 'game', array('control' => 'jform', 'load_data' => $loadData));

		if (empty($form)) {
			return false;
		}

		return $form;
	}

	/**
	 * Method to get the data that should be injected in the form.
	 *
	 * @return	mixed	The data for the form.
	 * @since	1.6
	 */
	protected function loadFormData()
	{
		// Check the session for previously entered form data.
		$data = JFactory::getApplication()->getUserState('com_draughts.edit.game.data', array());

		if (empty($data)) {
			$data = $this->getItem();
		}

		return $data;
	}

	/**
	 * Method to get a single record.
	 *
	 * @param	integer	The id of the primary key.
	 *
	 * @return	mixed	Object on success, false on failure.
	 * @since	1.6
	 */
	public function getItem($pk = null)
	{
		if ($item = parent::getItem($pk)) {

			//Do any procesing on fields here if needed

		}

		return $item;
	}

	/**
	 * Prepare and sanitise the table prior to saving.
	 *
	 * @since	1.6
	 */
	protected function prepareTable($table)
	{
		jimport('joomla.filter.output');

		if (empty($table->id)) {

			// Set ordering to the last item if not set
			if (@$table->ordering === '') {
				$db = JFactory::getDbo();
				$db->setQuery('SELECT MAX(ordering) FROM #__draughts_games');
				$max = $db->loadResult();
				$table->ordering = $max+1;
			}

		}
	}

	/**
	 * Method to save the form data.
	 *
	 * @param   array  $data  The form data.
	 *
	 * @return  boolean  True on success.
	 *
	 * @since   1.6
	 */
	public function save($data)
	{
		if(($position = $this->form_validate_draughts_position($data['position'])) === false):
			$this->setError(JText::_('COM_DRAUGHTS_FORM_SUBMIT_ERROR_POSITION'));
			return false;
		endif;
		if($this->form_validate_draughts_movement($data['movement'], $position) === false):
			// Error message was created by above function
			return false;
		endif;
		return parent::save($data);
	}

	/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
	 * 
	 * Custom functions for Form Validation
	 *
	 *  * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

	private function form_validate_draughts_position($position)
	{

		$position = preg_replace('/[^0-9wbpk:,]/i', '', $position);
		$position = strtolower($position);
		// Basic Validation
		if(preg_match('/[,]{2,}/', $position) || !preg_match_all('/wp:([0-9,]+)|wk:([0-9,]+)|bp:([0-9,]+)|bk:([0-9,]+)/i', $position, $pieces) || $position !== implode('', $pieces[0])):
			return false;
		endif;
		// More Validation
		$json = array();
		foreach($pieces[0] as $piece)
		{
			$piece = explode(':', $piece);
			array_push($json, '"' . $piece[0].'":["' . str_replace(',', '","', trim($piece[1], ',')) . '"]');
		}
		$json = '{'.implode(',', $json).'}';
		$position = json_decode($json);
		// Checking for Position & Piece Count
		$white_pieces = 0;
		$black_pieces = 0;
		foreach($position as $type => $pieces)
		{
			foreach($pieces as $piece)
			{
				substr($type, 0, 1) === 'w' ? $white_pieces ++ : $black_pieces ++;
				if($piece === 0 || $piece > 50 || $white_pieces > 20 || $black_pieces > 20):
					return false;
				endif;
			}
		}
		return $json;

	}

	private function form_validate_draughts_movement($moves, $position)
	{

		if(empty($moves))
			return true; // For moves return '{}';
		$position = json_decode($position);
		if(is_null($position)) // invalid position input
		{
			// Error 1: Invalid Position
			$this->setError(JText::_('COM_DRAUGHTS_FORM_SUBMIT_ERROR_MOVEMENT_1'));
			return false;
		}
		// Initiliaze Board & Checking for Position
		$_board = array();
		for($x = 1; $x <= 10; $x++)
		{
			for($y = 1; $y <= 10; $y++)
			{
				$_board[$x][$y] = false;
			}
		}
		foreach($position as $type => $pieces)
		{
			foreach($pieces as $piece)
			{
				$coordinat = $this->coordinat($piece);
				$piece_properties = new StdClass;
				$piece_properties->piece_type = $type;
				$piece_properties->piece_number = $piece;
				$_board[$coordinat->x][$coordinat->y] = $piece_properties;
			}
		}
		// Initiliaze Pieces
		$_pieces = array();
		foreach($position as $type => $pieces)
		{
			foreach($pieces as $position)
			{
				$piece = new StdClass;
				$piece->moves = array();
				$piece->crowned = (int) in_array($type, array('wk', 'bk'));
				$piece->eaten = 0;
				$_pieces[$position] = $piece;
			}
		}
		// Initiliaze Given Moves
		$_moves = explode(',', $moves);
		// Analyze Moves
		$number_of_moves = 0;
		$turn_previous = null;
		foreach($_moves as $move)
		{
			if(!preg_match('/([0-9]{1,2})-([0-9]{1,2})\??([0-9&]+)?/', $move, $tmp))
			{
				// Error 02: Invalid Move - Bad Move Format
				$this->setError(JText::_('COM_DRAUGHTS_FORM_SUBMIT_ERROR_MOVEMENT_2'));
				return false;
			}
			$move = new StdClass;
			$move->start = sprintf('%02d', $tmp[1]);
			$move->finish = sprintf('%02d', $tmp[2]);
			if($move->start == 0 || $move->start > 50 || $move->finish == 0 || $move->finish > 50) // invalid move input
			{
				// Error 03: Invalid Move - Wrong Square Number
				$this->setError(JText::_('COM_DRAUGHTS_FORM_SUBMIT_ERROR_MOVEMENT_3'));
				return false;
			}
			if(array_key_exists(3, $tmp))
			{
				$move->milestone = explode('&', $tmp[3]);
				foreach($move->milestone as &$milestone)
				{
					if($milestone == 0 || $milestone > 50 || isset($milestone[2])) // invalid move helping input
					{
						// Error 04: Invalid Move - Bad Move Helping
						$this->setError(JText::_('COM_DRAUGHTS_FORM_SUBMIT_ERROR_MOVEMENT_4'));
						return false;
					}
					if(!isset($milestone[1]))
						$milestone = sprintf('%02d', $milestone);
				}
			}
			else
			{
				$move->milestone = array();
			}
			unset($tmp);
			$coordinat = new StdClass;
			$coordinat->start = $this->coordinat($move->start);
			$coordinat->finish = $this->coordinat($move->finish);
			// Checking for turn
			$turn_current = substr($_board[$coordinat->start->x][$coordinat->start->y]->piece_type, 0, 1);
			$game_start = true;
			for($square_number = 1; $game_start && $square_number <= 20; $square_number++)
			{
				$square_number = sprintf('%02d', $square_number);
				$square_coordinat = $this->coordinat($square_number);
				if($_board[$square_coordinat->x][$square_coordinat->y] === false || substr($_board[$square_coordinat->x][$square_coordinat->y]->piece_type, 0, 1) !== 'b')
				{
					$game_start = false;
				}
			}
			for($square_number = 21; $game_start && $square_number <= 30; $square_number++)
			{
				$square_number = sprintf('%02d', $square_number);
				$square_coordinat = $this->coordinat($square_number);
				if($_board[$square_coordinat->x][$square_coordinat->y] !== false)
				{
					$game_start = false;
				}
			}
			for($square_number = 31; $game_start && $square_number <= 50; $square_number++)
			{
				$square_number = sprintf('%02d', $square_number);
				$square_coordinat = $this->coordinat($square_number);
				if($_board[$square_coordinat->x][$square_coordinat->y] === false || substr($_board[$square_coordinat->x][$square_coordinat->y]->piece_type, 0, 1) !== 'w')
				{
					$game_start = false;
				}
			}
			if($game_start && $turn_current !== 'w')
			{
				// Error 05: Invalid Move - Wrong Starting Turn
				$this->setError(JText::_('COM_DRAUGHTS_FORM_SUBMIT_ERROR_MOVEMENT_5'));
				return false;
			}
			if(is_null($turn_previous)) // initiliaze turn
			{
				$turn_previous = substr($_board[$coordinat->start->x][$coordinat->start->y]->piece_type, 0, 1);
			}
			else // 2nd turns
			{
				if($turn_previous === $turn_current) // invalid move - wrong turn
				{
					// Error 06: Invalid Move - Wrong Turn
					$this->setError(JText::_('COM_DRAUGHTS_FORM_SUBMIT_ERROR_MOVEMENT_6'));
					return false;
				}
				$turn_previous = $turn_current;
			}
			// Man or King & Normal Move
			if(($is_normal_move = $this->is_normal_move($_board, $coordinat->start, $coordinat->finish)) === true)
			{
				for($square_number = 1; $square_number <= 50; $square_number++)
				{
					$square_number = sprintf('%02d', $square_number);
					$initial_position = $this->coordinat($square_number);
					if($_board[$initial_position->x][$initial_position->y] === false || substr($_board[$initial_position->x][$initial_position->y]->piece_type, 0, 1) !== $turn_current || $this->eats($_board, $initial_position) === false)
					{
						continue;
					}
					if($initial_position !== $coordinat->start)
					{
						// Error 07: Invalid Move - Capturing was skipped
						$this->setError(JText::_('COM_DRAUGHTS_FORM_SUBMIT_ERROR_MOVEMENT_7'));
						return false;
					}
				}
				$move_details = new StdClass;
				$move_details->position = $move->finish;
				$move_details->wait = $number_of_moves;
				if(($count = count($_pieces[$_board[$coordinat->start->x][$coordinat->start->y]->piece_number]->moves)) > 0)
				{
					for($i = 0; $i < $count; $i++)
					{
						$move_details->wait -= $_pieces[$_board[$coordinat->start->x][$coordinat->start->y]->piece_number]->moves[$i]->wait;
					}
					$move_details->wait -= $count;
				}
				// Save to Output
				array_push($_pieces[$_board[$coordinat->start->x][$coordinat->start->y]->piece_number]->moves, $move_details);
				// Update Board
				$piece_properties = new StdClass;
				$piece_properties->piece_type = $_board[$coordinat->start->x][$coordinat->start->y]->piece_type;
				$piece_properties->piece_number = $_board[$coordinat->start->x][$coordinat->start->y]->piece_number;
				$_board[$coordinat->finish->x][$coordinat->finish->y] = $piece_properties;
				$_board[$coordinat->start->x][$coordinat->start->y] = false;
				// Count Moves
				$number_of_moves++;
				// Become King
				if($_board[$coordinat->finish->x][$coordinat->finish->y]->piece_type === 'wp' && in_array($move->finish, array(1,2,3,4,5)) || $_board[$coordinat->finish->x][$coordinat->finish->y]->piece_type === 'bp' && in_array($move->finish, array(46,47,48,49,50)))
				{
					$_board[$coordinat->finish->x][$coordinat->finish->y]->piece_type = str_replace('p', 'k', $_board[$coordinat->finish->x][$coordinat->finish->y]->piece_type);
					$_pieces[$_board[$coordinat->finish->x][$coordinat->finish->y]->piece_number]->crowned = $number_of_moves;
				}
			}
			// Man or King & Eat
			elseif($is_normal_move === false)
			{
				// Search possible eats & paths
				$branches = array();
				for($square_number = 1; $square_number <= 50; $square_number++)
				{
					$square_number = sprintf('%02d', $square_number);
					$initial_position = $this->coordinat($square_number);
					if($_board[$initial_position->x][$initial_position->y] === false || substr($_board[$initial_position->x][$initial_position->y]->piece_type, 0, 1) !== $turn_current || $this->eats($_board, $initial_position) === false)
						continue;
					$possible_branches = array();
					array_push($possible_branches, 'p'.$square_number);
					for($branch_id = 0; $branch_id < count($possible_branches); $branch_id++)
					{
						$branch = $possible_branches[$branch_id];
						$existing_eats = array();
						if(preg_match_all('/e([0-9]{2})/', $branch, $previous_eats)):
							foreach($previous_eats[1] as $eats)
							{
								array_push($existing_eats, $this->coordinat($eats));
							}
						endif;
						$last_position = $this->coordinat(substr($branch, -2));
						$board = $_board;
						$piece_properties = new StdClass;
						$piece_properties->piece_type = $board[$initial_position->x][$initial_position->y]->piece_type;
						$board[$last_position->x][$last_position->y] = $piece_properties;
						$eats = $this->eats($board, $last_position, $existing_eats);
						if($last_position != $initial_position)
							$board[$initial_position->x][$initial_position->y] = false;
						while($eats !== false)
						{
							if(in_array($_board[$initial_position->x][$initial_position->y]->piece_type, array('wp', 'bp')))
							{ // Man
								$new_branch = new StdClass;
								$new_branch->x = 2 * $eats->x - $last_position->x;
								$new_branch->y = 2 * $eats->y - $last_position->y;
								array_push($possible_branches, $branch.'e'.$this->coordinat2square_number($eats).'p'.$this->coordinat2square_number($new_branch));
							}
							else // King
							{
								$board_temp = $board;
								$k = 1;
								$new_branch = new StdClass;
								$new_branch->x = $eats->x + ($eats->x - $last_position->x > 0 ? 1 : -1) * $k;
								$new_branch->y = $eats->y + ($eats->y - $last_position->y > 0 ? 1 : -1) * $k;
								while(isset($board_temp[$new_branch->x][$new_branch->y]) && $board_temp[$new_branch->x][$new_branch->y] === false)
								{
									array_push($possible_branches, $branch.'e'.$this->coordinat2square_number($eats).'p'.$this->coordinat2square_number($new_branch));
									$k++;
									$new_branch = new StdClass;
									$new_branch->x = $eats->x + ($eats->x - $last_position->x > 0 ? 1 : -1) * $k;
									$new_branch->y = $eats->y + ($eats->y - $last_position->y > 0 ? 1 : -1) * $k;
								}
							}
							array_push($existing_eats, $eats);
							$eats = $this->eats($board, $last_position, $existing_eats);
						}
					}
					// Search longest path
					$max_lenght = max(array_map('strlen', $possible_branches));
					foreach($possible_branches as $key => $branch)
					{
						if(strlen($branch) < $max_lenght):
							unset($possible_branches[$key]);
							continue;
						endif;
						array_push($branches, $branch);
					}
				}
				// Search correct path
				foreach($branches as $key => $branch)
				{
					if(strpos($branch, 'p'.$move->start) === false):
						unset($branches[$key]);
					endif;
					foreach($move->milestone as $milestone)
					{
						if(strpos($branch, 'p'.$milestone) === false):
							unset($branches[$key]);
							break;
						endif;
					}
					if(strpos($branch, 'p'.$move->finish) === false):
						unset($branches[$key]);
					endif;
				}
				if(count($branches) !== 1) // irregular move
				{
					// Error 08: Invalid Move - Wrong Capturing;
					$this->setError(JText::_('COM_DRAUGHTS_FORM_SUBMIT_ERROR_MOVEMENT_8'));
					return false;
				}
				$branches = str_split(implode($branches), 3);
				// Move
				foreach($branches as $key => $branch)
				{
					if($key === 0): // starting position
						continue;
					endif;
					if(substr($branch, 0, 1) === 'p') // move
					{
						$move_details = new StdClass;
						$move_details->position = substr($branch, 1);
						if($key === 2) // first move
						{
							$move_details->wait = $number_of_moves;
							if(($count = count($_pieces[$_board[$coordinat->start->x][$coordinat->start->y]->piece_number]->moves)) > 0)
							{
								for($i = 0; $i < $count; $i++)
								{
									$move_details->wait -= $_pieces[$_board[$coordinat->start->x][$coordinat->start->y]->piece_number]->moves[$i]->wait;
								}
								$move_details->wait -= $count;
							}
						}
						else // 2nd moves
						{
							$move_details->wait = 0;
						}
						// Save to Output
						array_push($_pieces[$_board[$coordinat->start->x][$coordinat->start->y]->piece_number]->moves, $move_details);
						// Count Moves
						$number_of_moves++;
					}
				}
				// Eat
				foreach($branches as $key => $branch)
				{
					if(substr($branch, 0, 1) === 'e') // eat
					{
						// Save to Output
						$eat_coordinat = $this->coordinat(substr($branch, 1));
						$eaten = $number_of_moves;
						if(($count = count($_pieces[$_board[$eat_coordinat->x][$eat_coordinat->y]->piece_number]->moves)) > 0)
						{
							for($i = 0; $i < $count; $i++)
							{
								$eaten -= $_pieces[$_board[$eat_coordinat->x][$eat_coordinat->y]->piece_number]->moves[$i]->wait;
							}
							$eaten -= $count;
						}
						$_pieces[$_board[$eat_coordinat->x][$eat_coordinat->y]->piece_number]->eaten = $eaten;
						// Update board
						$_board[$eat_coordinat->x][$eat_coordinat->y] = false;
					}
				}
				// Update Board
				$piece_properties = new StdClass;
				$piece_properties->piece_type = $_board[$coordinat->start->x][$coordinat->start->y]->piece_type;
				$piece_properties->piece_number = $_board[$coordinat->start->x][$coordinat->start->y]->piece_number;
				$_board[$coordinat->finish->x][$coordinat->finish->y] = $piece_properties;
				$_board[$coordinat->start->x][$coordinat->start->y] = false;
				// Become King
				if($_board[$coordinat->finish->x][$coordinat->finish->y]->piece_type === 'wp' && in_array($move->finish, array(1,2,3,4,5)) || $_board[$coordinat->finish->x][$coordinat->finish->y]->piece_type === 'bp' && in_array($move->finish, array(46,47,48,49,50)))
				{
					$_board[$coordinat->finish->x][$coordinat->finish->y]->piece_type = str_replace('p', 'k', $_board[$coordinat->finish->x][$coordinat->finish->y]->piece_type);
					$_pieces[$_board[$coordinat->finish->x][$coordinat->finish->y]->piece_number]->crowned = $number_of_moves;
				}
			}
			// Error @ Man or King Move
			else
			{
				// Error message was created in function is_normal_move
				return false;
			}
		}
		return true; // for Moves return json_encode($_pieces);

	}

	private function eats($_board, $coordinat, $existing_eats = array())
	{

		$piece_properties = new StdClass;
		$piece_properties->piece_type = $_board[$coordinat->x][$coordinat->y]->piece_type;
		// for Man
		if(in_array($piece_properties->piece_type, array('wp', 'bp')))
		{
			$enemies = ($piece_properties->piece_type === 'wp' ? array('bp', 'bk') : array('wp', 'wk'));
			foreach(array(1, -1) as $x)
			{
				foreach(array(1, -1) as $y)
				{
					if(isset($_board[$coordinat->x + $x][$coordinat->y + $y]) && isset($_board[$coordinat->x + $x][$coordinat->y + $y]->piece_type) && in_array($_board[$coordinat->x + $x][$coordinat->y + $y]->piece_type, $enemies) && isset($_board[$coordinat->x + 2 * $x][$coordinat->y + 2 * $y]) && is_bool($_board[$coordinat->x + 2 * $x][$coordinat->y + 2 * $y]))
					{
						$eats = new StdClass;
						$eats->x = $coordinat->x + $x;
						$eats->y = $coordinat->y + $y;
						if(!in_array($eats, $existing_eats))
						{
							return $eats;
						}
					}
				}
			}
		}
		// for King
		if(in_array($piece_properties->piece_type, array('wk', 'bk')))
		{
			$enemies = ($piece_properties->piece_type === 'wk' ? array('bp', 'bk') : array('wp', 'wk'));
			$directions = array(array(1,1), array(1,-1), array(-1,1), array(-1,-1));
			foreach($directions as &$direction)
			{
				for($i = 1; $i <= 8; $i++)
				{
					$x = $direction[0] * $i;
					$y = $direction[1] * $i;
					if(isset($_board[$coordinat->x + $x][$coordinat->y + $y]) && isset($_board[$coordinat->x + $x][$coordinat->y + $y]->piece_type) && !in_array($_board[$coordinat->x + $x][$coordinat->y + $y]->piece_type, $enemies))
					{
						unset($direction);
						$i = 9;
					}
					elseif(isset($_board[$coordinat->x + $x][$coordinat->y + $y]) && isset($_board[$coordinat->x + $x][$coordinat->y + $y]->piece_type) && in_array($_board[$coordinat->x + $x][$coordinat->y + $y]->piece_type, $enemies) && (!isset($_board[$coordinat->x + $x + $direction[0]][$coordinat->y + $y + $direction[1]]) || !is_bool($_board[$coordinat->x + $x + $direction[0]][$coordinat->y + $y + $direction[1]])))
					{
						unset($direction);
						$i = 9;
					}
					elseif(isset($_board[$coordinat->x + $x][$coordinat->y + $y]) && isset($_board[$coordinat->x + $x][$coordinat->y + $y]->piece_type) && in_array($_board[$coordinat->x + $x][$coordinat->y + $y]->piece_type, $enemies) && isset($_board[$coordinat->x + $x + $direction[0]][$coordinat->y + $y + $direction[1]]) && is_bool($_board[$coordinat->x + $x + $direction[0]][$coordinat->y + $y + $direction[1]]))
					{
						$eats = new StdClass;
						$eats->x = $coordinat->x + $x;
						$eats->y = $coordinat->y + $y;
						if(!in_array($eats, $existing_eats))
						{
							return $eats;
						}
					}
				}
			}
		}
		return false;

	}

	/*
	 *
	 * name: is_normal_move
	 * @param $_board, $start, $finish
	 * @return Returns true, if move is valid, false otherwise 
	 *
	 */

	private function is_normal_move($_board, $start, $finish)
	{

		if(in_array($_board[$start->x][$start->y]->piece_type, array('wp', 'bp')))
		{
			if($this->eats($_board, $start) !== false) // eat
				return false;
			if(abs($start->x - $finish->x) != 1 || abs($start->y - $finish->y) != 1): // invalid move input
				$this->setError(JText::_('COM_DRAUGHTS_FORM_SUBMIT_ERROR_MOVEMENT_9'));
				return;
			endif;
			if($_board[$start->x][$start->y]->piece_type === 'wp' && $start->y < $finish->y || $_board[$start->x][$start->y]->piece_type === 'bp' && $start->y > $finish->y): // invalid move input - man backward
				$this->setError(JText::_('COM_DRAUGHTS_FORM_SUBMIT_ERROR_MOVEMENT_10'));
				return;
			endif;
		}
		elseif(in_array($_board[$start->x][$start->y]->piece_type, array('wk', 'bk')))
		{
			if($this->eats($_board, $start) !== false) // eat
				return false;
			if(abs($start->x - $finish->x) != abs($start->y - $finish->y)): // invalid move input - no diagonal
				$this->setError(JText::_('COM_DRAUGHTS_FORM_SUBMIT_ERROR_MOVEMENT_11'));
				return false;
			endif;
		}
		else
		{
			return false;
		}
		return true;

	}

	/*
	 *
	 * name: coordinat
	 * @param $square_number
	 * @return Returns coordinat of given square
	 *
	 */

	private function coordinat($square_number){

		$coordinat = new StdClass;
		$coordinat->x = 1;
		$coordinat->y = 1;
		if(($square_number - 1) % 10 < 5){
			$coordinat->x += (($square_number - 1) % 5 * 2 + 1);
		}else{
			$coordinat->x += (($square_number - 1) % 5 * 2);
		}
		$coordinat->y += floor(($square_number - 1) / 5);
		return $coordinat;

	}

	/*
	 *
	 * name: coordinat2square_number
	 * @param $coordinat
	 * @return Returns square number related to given coordinat
	 *
	 */

	private function coordinat2square_number($coordinat){

		$square_number = 0;
		$square_number += ($coordinat->y - 1) * 5;
		if($coordinat->x % 2 === 0)
		{
			$square_number += $coordinat->x / 2;
		}
		else
		{
			$square_number += ($coordinat->x + 1) / 2;
		}
		return sprintf('%02d', $square_number);

	}

	// Custom Function for Data Preparing

	private function convert_json($json)
	{

		return htmlspecialchars($json, ENT_QUOTES);

	}

}
