<?php

/**
 * @version     2.0.0
 * @package     com_draughts
 * @copyright   Copyright (C) 2015. All rights reserved.
 * @license     GNU General Public License version 2 or later; see LICENSE.txt
 * @author      Galaa <contact@galaa.mn> - http://galaa.mn
 */
defined('_JEXEC') or die;

jimport('joomla.application.component.modellist');

/**
 * Methods supporting a list of Draughts records.
 */
class DraughtsModelGames extends JModelList
{

	/**
	 * Constructor.
	 *
	 * @param    array    An optional associative array of configuration settings.
	 *
	 * @see        JController
	 * @since      1.6
	 */
	public function __construct($config = array())
	{
		if (empty($config['filter_fields']))
		{
			$config['filter_fields'] = array(
				'id', 'a.id',
                'ordering', 'a.ordering',
                'state', 'a.state',
                'created_by', 'a.created_by',
                'title', 'a.title',
                'category', 'a.category',
                'tag', 'a.tag',
                'position', 'a.position',
                'movement', 'a.movement',
                'game_date', 'a.game_date',
                'game_place', 'a.game_place',
                'white_player_name', 'a.white_player_name',
                'black_player_name', 'a.black_player_name',
                'winner', 'a.winner',
                'description', 'a.description',
			);
		}
		parent::__construct($config);
	}

	/**
	 * Method to auto-populate the model state.
	 *
	 * Note. Calling getState in this method will result in recursion.
	 *
	 * @since    1.6
	 */
	protected function populateState($ordering = null, $direction = null)
	{

		// Initialise variables.
		$app = JFactory::getApplication();

		// List state information
		$input = JFactory::getApplication()->input;
		$menuitemid = $input->getInt('Itemid');
		$menu = new JApplicationSite;
		$menu = $menu->getMenu();
		if ($menuitemid) {
			$menuparams = $menu->getParams( $menuitemid );
		}
		if(isset($menuparams)):
			$limit = $menuparams->get('list-view-limit', 20);
		endif;
		if(!isset($limit) || !$limit):
			$limit = $app->getUserStateFromRequest('global.list.limit', 'limit', $app->getCfg('list_limit'));
		endif;
		$this->setState('list.limit', $limit);

		$limitstart = $app->input->getInt('limitstart', 0);
		$this->setState('list.start', $limitstart);

		if ($list = $app->getUserStateFromRequest($this->context . '.list', 'list', array(), 'array'))
		{
			foreach ($list as $name => $value)
			{
				// Extra validations
				switch ($name)
				{
					case 'fullordering':
						$orderingParts = explode(' ', $value);

						if (count($orderingParts) >= 2)
						{
							// Latest part will be considered the direction
							$fullDirection = end($orderingParts);

							if (in_array(strtoupper($fullDirection), array('ASC', 'DESC', '')))
							{
								$this->setState('list.direction', $fullDirection);
							}

							unset($orderingParts[count($orderingParts) - 1]);

							// The rest will be the ordering
							$fullOrdering = implode(' ', $orderingParts);

							if (in_array($fullOrdering, $this->filter_fields))
							{
								$this->setState('list.ordering', $fullOrdering);
							}
						}
						else
						{
							$this->setState('list.ordering', $ordering);
							$this->setState('list.direction', $direction);
						}
						break;

					case 'ordering':
						if (!in_array($value, $this->filter_fields))
						{
							$value = $ordering;
						}
						break;

					case 'direction':
						if (!in_array(strtoupper($value), array('ASC', 'DESC', '')))
						{
							$value = $direction;
						}
						break;

					case 'limit':
						$limit = $value;
						break;

					// Just to keep the default case
					default:
						$value = $value;
						break;
				}

				$this->setState('list.' . $name, $value);
			}
		}

		// Receive & set filters
		if(isset($menuparams))
		{
			$categories = $menuparams->get('categories', array());
			foreach ($categories as $name => $value)
			{
				$this->setState('filter.menuparameter-categories-' . $name, $value);
			}
		}
		if(isset($categories) && empty($categories) && $filters = $app->getUserStateFromRequest($this->context . '.filter', 'filter', array(), 'array'))
		{
			foreach ($filters as $name => $value)
			{
				$this->setState('filter.' . $name, $value);
			}
		}

		$ordering = $app->input->get('filter_order');
		if (!empty($ordering))
		{
			$list             = $app->getUserState($this->context . '.list');
			$list['ordering'] = $app->input->get('filter_order');
			$app->setUserState($this->context . '.list', $list);
		}

		$orderingDirection = $app->input->get('filter_order_Dir');
		if (!empty($orderingDirection))
		{
			$list              = $app->getUserState($this->context . '.list');
			$list['direction'] = $app->input->get('filter_order_Dir');
			$app->setUserState($this->context . '.list', $list);
		}

		$list = $app->getUserState($this->context . '.list');

		if (empty($list['ordering']))
{
	$list['ordering'] = 'ordering';
}

if (empty($list['direction']))
{
	$list['direction'] = 'asc';
}

		$this->setState('list.ordering', $list['ordering']);
		$this->setState('list.direction', $list['direction']);
	}

	/**
	 * Build an SQL query to load the list data.
	 *
	 * @return    JDatabaseQuery
	 * @since    1.6
	 */
	protected function getListQuery()
	{
		// Create a new query object.
		$db    = $this->getDbo();
		$query = $db->getQuery(true);

		// Select the required fields from the table.
		$query
			->select(
				$this->getState(
					'list.select', 'DISTINCT a.*'
				)
			);

		$query->from('`#__draughts_games` AS a');

		// Join over the users for the checked out user.
		$query->select('uc.name AS editor');
		$query->join('LEFT', '#__users AS uc ON uc.id=a.checked_out');

		// Join over the created by field 'created_by'
		$query->join('LEFT', '#__users AS created_by ON created_by.id = a.created_by');
		// Join over the category 'category'
		$query->select('category.title AS category_title');
		$query->join('LEFT', '#__categories AS category ON category.id = a.category');

		if (!JFactory::getUser()->authorise('core.edit.state', 'com_draughts'))
		{
			$query->where('a.state = 1');
		}

		// Filter by search in title
		$search = $this->getState('filter.search');
		if (!empty($search))
		{
			if (stripos($search, 'id:') === 0)
			{
				$query->where('a.id = ' . (int) substr($search, 3));
			}
			else
			{
				$search = $db->Quote('%' . $db->escape($search, true) . '%');
				$query->where('( a.title LIKE '.$search.'  OR  a.game_place LIKE '.$search.'  OR  a.white_player_name LIKE '.$search.'  OR  a.black_player_name LIKE '.$search.'  OR  a.description LIKE '.$search.' )');
			}
		}

		//Filtering category
		$filter_category = $this->state->get("filter.category");
		if ($filter_category) {
			$query->where("a.category = '".$db->escape($filter_category)."'");
		}

		// Add the list ordering clause.
		$orderCol  = $this->state->get('list.ordering');
		$orderDirn = $this->state->get('list.direction');
		if ($orderCol && $orderDirn)
		{
			$query->order($db->escape($orderCol . ' ' . $orderDirn));
		}

		return $query;

	}

	public function getItems()
	{
		$items = parent::getItems();
		foreach($items as $item){

			if ( isset($item->category) ) {

				// Get the title of that particular template
				$title = DraughtsFrontendHelper::getCategoryNameByCategoryId($item->category);

				// Finally replace the data object with proper information
				$item->category = !empty($title) ? $title : $item->category;
			}

			if ( isset($item->tag) ) {
				// Catch the item tags (string with ',' coma glue)
				$tags = explode(",",$item->tag);

				$db = JFactory::getDbo();
				$namedTags = array(); // Cleaning and initalization of named tags array

				// Get the tag names of each tag id
				foreach ($tags as $tag) {

					$query = $db->getQuery(true);
					$query->select("title");
					$query->from('`#__tags`');
					$query->where( "id=" . intval($tag) );
					$db->setQuery($query);
					$row = $db->loadObjectList();

					// Read the row and get the tag name (title)
					if (!is_null($row)) {
						foreach ($row as $value) {
							if ( $value && isset($value->title) ) {
								$namedTags[] = trim($value->title);
							}
						}
					}

				}

				// Finally replace the data object with proper information
				$item->tag = !empty($namedTags) ? implode(', ',$namedTags) : $item->tag;
			}
			$item->winner = JText::_('COM_DRAUGHTS_GAMES_WINNER_OPTION_' . strtoupper($item->winner));
			// Movement Speed
			$app = JFactory::getApplication();
			$params = $app->getParams();
			$item->speed = $params->get('speed');

		}

		return $items;
	}

	/**
	 * Overrides the default function to check Date fields format, identified by
	 * "_dateformat" suffix, and erases the field if it's not correct.
	 */
	protected function loadFormData()
	{
		$app              = JFactory::getApplication();
		$filters          = $app->getUserState($this->context . '.filter', array());
		$error_dateformat = false;
		foreach ($filters as $key => $value)
		{
			if (strpos($key, '_dateformat') && !empty($value) && !$this->isValidDate($value))
			{
				$filters[$key]    = '';
				$error_dateformat = true;
			}
		}
		if ($error_dateformat)
		{
			$app->enqueueMessage(JText::_("COM_PRUEBA_SEARCH_FILTER_DATE_FORMAT"), "warning");
			$app->setUserState($this->context . '.filter', $filters);
		}

		return parent::loadFormData();
	}

	/**
	 * Checks if a given date is valid and in an specified format (YYYY-MM-DD)
	 *
	 * @param string Contains the date to be checked
	 *
	 */
	private function isValidDate($date)
	{
		return preg_match("/^(19|20)\d\d[-](0[1-9]|1[012])[-](0[1-9]|[12][0-9]|3[01])$/", $date) && date_create($date);
	}

}
